#![deny(missing_docs)]
#![deny(clippy::missing_docs_in_private_items)]
// SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
// SPDX-License-Identifier: BSD-2-Clause
//! Check whether example programs can be built.

use eyre::Result;

#[cfg(any(
    feature = "toml-facet030-unstable",
    feature = "toml-toml08",
    feature = "toml-toml09"
))]
mod cli;

#[cfg(any(
    feature = "toml-facet030-unstable",
    feature = "toml-toml08",
    feature = "toml-toml09"
))]
/// Do what we need to if compiled with TOML support.
mod with_or_without_toml {
    use camino::Utf8Path;
    use eyre::{Result, WrapErr as _, eyre};
    use itertools::Itertools as _;
    use log::{debug, info};
    use tempfile::Builder as TempBuilder;

    use check_build::Config;
    use check_build::parse;
    use check_build::prepare;
    use check_build::step;

    use crate::cli::{self, Mode};

    /// Parse the configuration file, build the programs, etc.
    fn run(cfg: &Config) -> Result<()> {
        let prog_cfg =
            parse::program_config(cfg).context("Could not parse the configuration file")?;
        debug!(
            "Parsed {count} program definition{plu}",
            count = prog_cfg.program().len(),
            plu = if prog_cfg.program().len() == 1 {
                ""
            } else {
                "s"
            }
        );

        for (name, prog) in prog_cfg
            .program()
            .iter()
            .sorted_unstable_by(|&(left, _), &(right, _)| left.cmp(right))
        {
            let tempd_obj = TempBuilder::new()
                .prefix("check-build.")
                .tempdir()
                .context("Could not create a temporary directory")?;
            let tempd = Utf8Path::from_path(tempd_obj.path()).ok_or_else(|| {
                eyre!(
                    "The temporary directory path {tempd} is not all valid UTF-8",
                    tempd = tempd_obj.path().display()
                )
            })?;
            debug!("Using {tempd} to build {name}");

            let benv = prepare::build_env(cfg, name, prog, tempd).with_context(|| {
                format!("Could not prepare a build environment for {name} in {tempd}")
            })?;

            if !step::prereq_satisfied(&benv)
                .with_context(|| format!("Could not check for the {name} prerequisites"))?
            {
                info!("The prerequisites for {name} were not satisfied, skipping");
                continue;
            }

            step::build(&benv).with_context(|| format!("Could not build {name}"))?;
            step::test(&benv).with_context(|| format!("Could not test {name}"))?;
            info!(
                "Successfully built and run {name} in {path}",
                path = benv.path()
            );
        }
        Ok(())
    }

    /// The main program - parse command-line options, do something about it.
    pub fn main() -> Result<()> {
        match cli::try_parse()? {
            Mode::Run(cfg) => run(&cfg),
        }
    }
}

#[cfg(all(
    not(feature = "toml-facet030-unstable"),
    not(feature = "toml-toml08"),
    not(feature = "toml-toml09")
))]
/// Bail out if compiled without TOML support.
mod with_or_without_toml {
    use eyre::{Result, bail};

    /// The main program - parse command-line options, do something about it.
    pub fn main() -> Result<()> {
        bail!("No TOML parsing support at compile time")
    }
}

/// The main program - parse command-line options, do something about it.
fn main() -> Result<()> {
    with_or_without_toml::main()
}

#[cfg(test)]
#[cfg(any(
    feature = "toml-facet030-unstable",
    feature = "toml-toml08",
    feature = "toml-toml09"
))]
#[expect(clippy::print_stdout, reason = "unit tests")]
mod tests {
    use std::env;
    use std::io::ErrorKind;
    use std::process::{Command, Stdio};
    use std::sync::LazyLock;

    use camino::{Utf8Path, Utf8PathBuf};
    use eyre::{Result, WrapErr as _, bail, eyre};

    /// Find the executable to test.
    ///
    /// # Errors
    ///
    /// Propagate errors from `env::current_exe()`, `Utf8PathBuf::name()`, etc.
    /// Fail if the executable file could not be found.
    pub fn get_exe_path() -> Result<Utf8PathBuf> {
        static PATH: LazyLock<Result<Utf8PathBuf>> = LazyLock::new(|| {
            let current = Utf8PathBuf::from_path_buf(
                env::current_exe().context("Could not get the current executable file's path")?,
            )
            .map_err(|path| {
                eyre!(
                    "Could not represent {path} as valid UTF-8",
                    path = path.display()
                )
            })?;
            let exe_dir = {
                let basedir = current
                    .parent()
                    .ok_or_else(|| eyre!("Could not get the parent directory of {current}"))?;
                if basedir
                    .file_name()
                    .ok_or_else(|| eyre!("Could not get the base name of {basedir}"))?
                    == "deps"
                {
                    basedir
                        .parent()
                        .ok_or_else(|| eyre!("Could not get the parent directory of {basedir}"))?
                } else {
                    basedir
                }
            };
            let exe_path = exe_dir.join("check-build");
            if !exe_path.is_file() {
                bail!("Not a regular file: {exe_path}");
            }
            Ok(exe_path)
        });

        match *PATH {
            Ok(ref res) => Ok(res.clone()),
            Err(ref err) => bail!("Could not determine the path to the test program: {err}"),
        }
    }

    #[test]
    fn python_functional() -> Result<()> {
        let func = {
            let relpath: &Utf8Path = "../python/tests/functional".into();
            match relpath.canonicalize_utf8() {
                Ok(path) => path,
                Err(err) if err.kind() == ErrorKind::NotFound => {
                    println!("No {relpath} directory, skipping");
                    return Ok(());
                }
                Err(err) => return Err(err).with_context(|| relpath.to_string()),
            }
        };

        let python3 = "python3";
        if let Err(err) = Command::new(python3)
            .arg("--version")
            .stdin(Stdio::null())
            .stdout(Stdio::piped())
            .stderr(Stdio::piped())
            .output()
        {
            if err.kind() == ErrorKind::NotFound {
                println!("No {python3}, skipping");
                return Ok(());
            }
            return Err(err).with_context(|| format!("{python3} --version"));
        }

        let exe = get_exe_path()?;
        if !Command::new(python3)
            .args(["-B", "-u", "-m", "check_build_test", "--", exe.as_str()])
            .env("PYTHONPATH", func.as_str())
            .stdin(Stdio::null())
            .stdout(Stdio::inherit())
            .stderr(Stdio::inherit())
            .status()
            .with_context(|| format!("Could not run {python3}"))?
            .success()
        {
            bail!("The Python functional test failed for {exe}");
        }
        Ok(())
    }
}
