// SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
// SPDX-License-Identifier: BSD-2-Clause
//! Perform the actual clean, build, and test steps.

use std::process::Command;

use camino::Utf8PathBuf;
use eyre::WrapErr as _;
use log::{debug, error, info, warn};
use which::Error as WhichError;

use crate::defs::Error;
use crate::prepare::BuildEnv;

/// Check whether a program's prerequisites are satisifed.
///
/// # Errors
///
/// [`Error::Which`] on failure to look for a program (other than the program not existing).
#[inline]
pub fn prereq_satisfied(benv: &BuildEnv<'_>) -> Result<bool, Error> {
    debug!(
        "Checking for the prerequisites of {name}",
        name = benv.name()
    );

    if let Some(prereq) = benv.prog().prerequisites() {
        let progs = prereq.programs();
        if !progs.is_empty() {
            debug!(
                "Checking for programs: {programs}",
                programs = shell_words::join(progs)
            );
            for prog in progs {
                match which::which(prog) {
                    Ok(_) => (),
                    Err(WhichError::CannotFindBinaryPath) => {
                        error!(
                            "The {prog} prerequisite for {name} was not found in the search path",
                            name = benv.name()
                        );
                        return Ok(false);
                    }
                    Err(err) => {
                        return Err(err)
                            .with_context(|| {
                                format!("Could not look for {prog} in the search path")
                            })
                            .map_err(Error::Which);
                    }
                }
            }
        }
    }

    Ok(true)
}

/// Run some commands within a build environment.
///
/// # Errors
///
/// [`Error::CommandRun`] if a build command could not be executed at all.
/// [`Error::CommandFailed`] if a build command failed.
fn run_commands(benv: &BuildEnv<'_>, tag: &str, commands: &[Vec<String>]) -> Result<(), Error> {
    for cmd in commands {
        if let Some((cmd_prog, rest)) = cmd.split_first() {
            debug!(
                "- running `{cmd}` in {path}",
                cmd = shell_words::join(cmd),
                path = benv.path()
            );
            let res = Command::new(cmd_prog)
                .args(rest)
                .current_dir(benv.path())
                .status()
                .map_err(|err| {
                    Error::CommandRun(
                        benv.name().to_owned(),
                        tag.to_owned(),
                        shell_words::join(cmd),
                        err,
                    )
                })?;
            if !res.success() {
                return Err(Error::CommandFailed(
                    benv.name().to_owned(),
                    tag.to_owned(),
                    shell_words::join(cmd),
                    res.code().unwrap_or(-1),
                ));
            }
        } else {
            warn!(
                "Empty {tag} command specified for {name}",
                name = benv.name()
            );
        }
    }
    Ok(())
}

/// Build a single program in a prepared build environment.
///
/// # Errors
///
/// [`Error::CommandRun`] if a build command could not be executed at all.
/// [`Error::CommandFailed`] if a build command failed.
/// [`Error::NotBuilt`] if the executable file was not found after running
/// all the build commands.
#[inline]
pub fn build(benv: &BuildEnv<'_>) -> Result<Utf8PathBuf, Error> {
    info!(
        "Building {name} in {path}",
        name = benv.name(),
        path = benv.path()
    );
    run_commands(benv, "build", benv.prog().commands().build())?;

    let executable = benv.path().join(benv.prog().executable());
    if !executable.is_file() {
        return Err(Error::NotBuilt(
            benv.name().to_owned(),
            executable.to_string(),
        ));
    }
    Ok(executable)
}

/// Test a single program in a prepared build environment.
///
/// # Errors
///
/// [`Error::CommandRun`] if a build command could not be executed at all.
/// [`Error::CommandFailed`] if a build command failed.
/// all the build commands.
#[inline]
pub fn test(benv: &BuildEnv<'_>) -> Result<(), Error> {
    info!(
        "Testing {name} in {path}",
        name = benv.name(),
        path = benv.path()
    );
    run_commands(benv, "test", benv.prog().commands().test())?;
    Ok(())
}

#[cfg(test)]
#[expect(clippy::panic_in_result_fn, reason = "unit tests")]
mod tests {
    use camino::Utf8PathBuf;
    use eyre::{Result, WrapErr as _, bail, eyre};
    use tempfile::TempDir;

    use crate::defs::Error;
    use crate::prepare;

    fn make_tempdir() -> Result<(TempDir, Utf8PathBuf)> {
        let tempd_obj = tempfile::tempdir().context("tempd")?;
        let tempd = Utf8PathBuf::from_path_buf(tempd_obj.path().to_path_buf())
            .map_err(|path| eyre!("tempd utf8 {path}", path = path.display()))?;
        Ok((tempd_obj, tempd))
    }

    #[test]
    fn touch_twice_bad() -> Result<()> {
        let (_tempd_obj, tempd) = make_tempdir()?;
        let work = tempd.join("work");

        let name = "touch-twice-bad";
        let (cfg, prog) =
            prepare::build_test_build_env(name, tempd.join("src")).context("prepare benv")?;
        let benv = prepare::build_env(&cfg, name, prog, &work).context("benv")?;

        assert!(!super::prereq_satisfied(&benv).context("prereq")?);
        assert_eq!(
            super::build(&benv).context("build")?,
            work.join(name).join("prog")
        );
        match super::test(&benv) {
            Ok(()) => bail!("Did not expect the test step to succeed"),
            Err(Error::CommandFailed(program, action, command, rcode))
                if program == "touch-twice-bad"
                    && action == "test"
                    && command == "test -d prog"
                    && rcode != 0_i32 =>
            {
                Ok(())
            }
            Err(err) => Err(err).context("Unexpected test failure"),
        }
    }

    #[test]
    fn touch_good() -> Result<()> {
        let (_tempd_obj, tempd) = make_tempdir()?;
        let work = tempd.join("work");

        let name = "touch-good";
        let (cfg, prog) =
            prepare::build_test_build_env(name, tempd.join("src")).context("prepare benv")?;
        let benv = prepare::build_env(&cfg, name, prog, &work).context("benv")?;

        assert!(super::prereq_satisfied(&benv).context("prereq")?);
        assert_eq!(
            super::build(&benv).context("build")?,
            work.join(name).join("prog")
        );
        super::test(&benv).context("test")
    }
}
