// SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
// SPDX-License-Identifier: BSD-2-Clause
//! Parse the configuration file that describes the programs to build.

use std::collections::HashMap;

#[cfg(any(
    feature = "toml-facet030-unstable",
    feature = "toml-toml08",
    feature = "toml-toml09"
))]
mod p_toml;

#[cfg(any(
    feature = "toml-facet030-unstable",
    feature = "toml-toml08",
    feature = "toml-toml09"
))]
pub use p_toml::program_config;

#[cfg(test)]
use std::sync::LazyLock;

#[cfg(feature = "facet030-unstable")]
use facet030::Facet;

#[cfg(feature = "serde")]
use serde_derive::Deserialize;

/// The conditions that need to be satisfied for a program to be built and run.
#[derive(Debug)]
#[cfg_attr(feature = "facet030-unstable", derive(Facet))]
#[cfg_attr(feature = "serde", derive(Deserialize))]
pub struct Prerequisites {
    /// The programs that need to be present in the search path.
    programs: Vec<String>,
}

impl Prerequisites {
    /// The programs that need to be present in the search path.
    #[inline]
    #[must_use]
    pub fn programs(&self) -> &[String] {
        &self.programs
    }
}

/// The commands to execute for each step.
#[derive(Debug)]
#[cfg_attr(feature = "facet030-unstable", derive(Facet))]
#[cfg_attr(feature = "serde", derive(Deserialize))]
pub struct Commands {
    /// The commands to run to build the program.
    build: Vec<Vec<String>>,

    /// The commands to run to test the program.
    test: Vec<Vec<String>>,
}

impl Commands {
    /// The commands to run to build the program.
    #[inline]
    #[must_use]
    pub fn build(&self) -> &[Vec<String>] {
        &self.build
    }

    /// The commands to run to test the program.
    #[inline]
    #[must_use]
    pub fn test(&self) -> &[Vec<String>] {
        &self.test
    }
}

/// The definition of a single program to build and test.
#[derive(Debug)]
#[cfg_attr(feature = "facet030-unstable", derive(Facet))]
#[cfg_attr(feature = "serde", derive(Deserialize))]
pub struct Program {
    /// The name of the executable file generated by the build step.
    executable: String,

    /// The conditions that need to be satisfied for a program to be built and run.
    prerequisites: Option<Prerequisites>,

    /// The commands to execute for each step.
    commands: Commands,
}

impl Program {
    /// The name of the executable file generated by the build step.
    #[inline]
    #[must_use]
    pub fn executable(&self) -> &str {
        &self.executable
    }

    /// The conditions that need to be satisfied for a program to be built and run.
    #[inline]
    #[must_use]
    pub const fn prerequisites(&self) -> Option<&Prerequisites> {
        self.prerequisites.as_ref()
    }

    /// The commands to execute for each step.
    #[inline]
    #[must_use]
    pub const fn commands(&self) -> &Commands {
        &self.commands
    }
}

/// The configuration file that describes the programs to run.
#[derive(Debug)]
#[cfg_attr(feature = "facet030-unstable", derive(Facet))]
#[cfg_attr(feature = "serde", derive(Deserialize))]
pub struct ProgramsConfig {
    /// The metadata about the configuration file format.
    #[cfg_attr(feature = "facet030-unstable", facet(rename = "mediaType"))]
    #[cfg_attr(feature = "serde", serde(rename = "mediaType"))]
    media_type: String,

    /// The definitions of the programs.
    program: HashMap<String, Program>,
}

impl ProgramsConfig {
    /// The metadata about the configuration file format.
    #[inline]
    #[must_use]
    pub fn media_type(&self) -> &str {
        &self.media_type
    }

    /// The definitions of the programs.
    #[inline]
    #[must_use]
    pub const fn program(&self) -> &HashMap<String, Program> {
        &self.program
    }
}

#[cfg(test)]
/// Simulate parsing a set of program definitions to use for testing.
#[inline]
#[must_use]
pub fn build_test_program_config() -> &'static ProgramsConfig {
    static PROGRAMS_CONFIG: LazyLock<ProgramsConfig> = LazyLock::new(|| ProgramsConfig {
        media_type: "vnd.net.ringlet.devel.check-build.config/programs.v0.1+toml".to_owned(),
        program: [
            (
                "touch-twice-bad".to_owned(),
                Program {
                    executable: "prog".to_owned(),
                    prerequisites: Some(Prerequisites {
                        programs: vec!["nonexistent".to_owned()],
                    }),
                    commands: Commands {
                        build: vec![vec!["touch".to_owned(), "prog".to_owned()]],
                        test: vec![vec!["test".to_owned(), "-d".to_owned(), "prog".to_owned()]],
                    },
                },
            ),
            (
                "touch-good".to_owned(),
                Program {
                    executable: "prog".to_owned(),
                    prerequisites: Some(Prerequisites {
                        programs: vec!["touch".to_owned(), "test".to_owned()],
                    }),
                    commands: Commands {
                        build: vec![vec!["touch".to_owned(), "prog".to_owned()]],
                        test: vec![vec!["test".to_owned(), "-f".to_owned(), "prog".to_owned()]],
                    },
                },
            ),
        ]
        .into(),
    });

    &PROGRAMS_CONFIG
}

#[cfg(test)]
#[expect(clippy::allow_attributes, reason = "feature matrix too complex")]
#[expect(clippy::panic_in_result_fn, reason = "unit tests")]
#[allow(unused_imports, reason = "feature matrix too complex")]
mod tests {
    use eyre::{Result, WrapErr as _, bail, eyre};

    #[cfg(feature = "facet030-unstable")]
    use facet_pretty030::FacetPretty as _;

    use crate::defs::Config;

    #[cfg(any(
        feature = "toml-facet030-unstable",
        feature = "toml-toml08",
        feature = "toml-toml09"
    ))]
    #[test]
    fn load_c() -> Result<()> {
        let cfg = Config::new("../c/programs.toml".into()).with_verbose(true);
        let res = super::program_config(&cfg).context("parse")?;
        assert_eq!(
            res.program()
                .get("triv-cc")
                .ok_or_else(|| eyre!("no triv-cc in {res:?}"))?
                .executable(),
            "triv"
        );
        assert!(
            res.program()
                .get("triv-make")
                .ok_or_else(|| eyre!("no triv-make in {res:?}"))?
                .prerequisites()
                .ok_or_else(|| eyre!("no triv-make.prerequisites in {res:?}"))?
                .programs()
                .contains(&"make".to_owned())
        );
        Ok(())
    }

    #[cfg(feature = "facet030-unstable")]
    #[test]
    fn facet_shape() -> Result<()> {
        let prog_config = super::build_test_program_config();
        let pretty = format!("{prog_config}", prog_config = prog_config.pretty());
        let doc = "The definitions of the programs";
        if !pretty.contains(doc) {
            bail!("No {doc} in:\n{pretty}");
        }
        Ok(())
    }
}
