// SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
// SPDX-License-Identifier: BSD-2-Clause
//! Parse a TOML file.

use std::fs;

use eyre::WrapErr as _;

#[cfg(feature = "facet030-unstable")]
use facet030::Facet;

use media_type_version::{Config as MTVConfig, Error as MTVError};

#[cfg(feature = "serde")]
use serde_derive::Deserialize;

#[cfg(not(feature = "toml-facet030-unstable"))]
use serde::de::DeserializeOwned;

use crate::defs::{Config, Error};

use super::ProgramsConfig;

#[cfg(feature = "toml-facet030-unstable")]
mod p_toml_facet {
    // SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
    // SPDX-License-Identifier: BSD-2-Clause
    //! Parse a TOML file using `facet_toml`.

    use eyre::eyre;
    use facet030::Facet;

    use crate::defs::Error;

    /// Parse a TOML document using [`facet-toml`].
    pub fn toml_parse_facet<'data, 'facet, T>(contents: &'data str) -> Result<T, Error>
    where
        'data: 'facet,
        T: Facet<'facet>,
    {
        facet_toml030::from_str(contents)
            .map_err(|err| Error::ConfigParse(eyre!("Could not parse the TOML contents: {err}")))
    }
}

#[cfg(not(feature = "toml-facet030-unstable"))]
mod p_toml_toml {
    // SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
    // SPDX-License-Identifier: BSD-2-Clause
    //! Parse a TOML file using `toml` and `serde`.

    use eyre::WrapErr as _;
    use serde::de::DeserializeOwned;

    use crate::defs::Error;

    /// Parse a TOML document using [`serde`] and [`toml`].
    pub fn toml_parse_serde<T>(contents: &str) -> Result<T, Error>
    where
        T: DeserializeOwned,
    {
        #[cfg(not(feature = "toml-toml09"))]
        let res = toml08::from_str(contents);
        #[cfg(feature = "toml-toml09")]
        let res = toml09::from_str(contents);

        res.context("Could not parse the TOML contents")
            .map_err(Error::ConfigParse)
    }
}

/// Just the format metadata to check the version.
#[derive(Debug)]
#[cfg_attr(feature = "facet030-unstable", derive(Facet))]
#[cfg_attr(feature = "serde", derive(Deserialize))]
struct MediaTypeOnly {
    /// The metadata about the configuration file format.
    #[cfg_attr(feature = "facet030-unstable", facet(rename = "mediaType"))]
    #[cfg_attr(feature = "serde", serde(rename = "mediaType"))]
    media_type: String,
}

impl MediaTypeOnly {
    /// The metadata about the configuration file format.
    #[inline]
    #[must_use]
    pub fn media_type(&self) -> &str {
        &self.media_type
    }
}

#[cfg(feature = "toml-facet030-unstable")]
/// Parse a TOML document using [`facet-toml`].
pub fn toml_parse<'data, 'facet, T>(contents: &'data str) -> Result<T, Error>
where
    'data: 'facet,
    T: Facet<'facet>,
{
    p_toml_facet::toml_parse_facet(contents)
}

#[cfg(not(feature = "toml-facet030-unstable"))]
/// Parse a TOML document using [`serde`] and [`toml`].
pub fn toml_parse<T>(contents: &str) -> Result<T, Error>
where
    T: DeserializeOwned,
{
    p_toml_toml::toml_parse_serde(contents)
}

/// Parse the TOML configuration file that describes the programs to run.
///
/// # Errors
///
/// [`Error::ConfigParse`] if the configuration file format is invalid.
/// [`Error::ConfigUnsupportedVersion`] if the configuration file declares
/// an unsupported format version.
#[inline]
pub fn program_config(cfg: &Config) -> Result<ProgramsConfig, Error> {
    let contents = fs::read_to_string(cfg.config()).map_err(Error::ConfigRead)?;
    let mtv_only: MediaTypeOnly = toml_parse(&contents)?;
    let mtv_cfg = MTVConfig::builder()
        .prefix("vnd.net.ringlet.devel.check-build.config/programs")
        .suffix("+toml")
        .build()
        .map_err(MTVError::into_owned_error)
        .context("Could not prepare to parse a mediaType string")
        .map_err(Error::Internal)?;
    let mtv_ver = media_type_version::extract(&mtv_cfg, mtv_only.media_type())
        .map_err(|err| Error::ConfigParseMediaType(err.into_owned_error()))?;
    if mtv_ver.major() != 0 {
        return Err(Error::ConfigUnsupportedVersion(
            mtv_ver.major(),
            mtv_ver.minor(),
        ));
    }

    toml_parse(&contents)
}
