// SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
// SPDX-License-Identifier: BSD-2-Clause
//! Parse command-line options for things and stuff.

use argh::FromArgs;
use camino::Utf8PathBuf;
use eyre::{Result, WrapErr as _};
use log::LevelFilter;
use simple_logger::SimpleLogger;

use check_build::defs::Config;

/// check whether some programs will build or not.
#[derive(Debug, FromArgs)]
struct Cli {
    /// the configuration file that describes the program to build and run
    #[argh(option, short = 'c')]
    config: Utf8PathBuf,

    /// verbose operation; display diagnostic output
    #[argh(switch, short = 'v')]
    verbose: bool,
}

/// What to do, what to do?
#[derive(Debug)]
pub enum Mode {
    /// Copy a directory tree, rename files, replace strings within them.
    Run(Config),
}

/// Initialize the logging subsystem provided by the `simple_logger` library.
fn setup_logging(verbose: bool) -> Result<()> {
    SimpleLogger::new()
        .without_timestamps()
        .with_level(if verbose {
            LevelFilter::Debug
        } else {
            LevelFilter::Info
        })
        .init()
        .context("Could not initialize the logger")
}

/// Parse the command-line arguments, determine the mode of operation.
///
/// # Errors
///
/// Propagate command-line parsing errors.
pub fn try_parse() -> Result<Mode> {
    let args: Cli = argh::from_env();
    setup_logging(args.verbose).context("Could not initialize the logging infrastructure")?;
    Ok(Mode::Run(
        Config::new(args.config).with_verbose(args.verbose),
    ))
}
