//! Common definitions for the `check-build` crate.

/*
 * SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
 * SPDX-License-Identifier: BSD-2-Clause
 */

use std::error::Error as StdError;
use std::fmt::{Display, Formatter, Result as FmtResult};
use std::io::Error as IoError;

use camino::{Utf8Path, Utf8PathBuf};
use eyre::Error as AnyError;
use media_type_version::OwnedError as MTVOwnedError;

/// An error that occurred during the doing of things.
#[derive(Debug)]
#[non_exhaustive]
#[expect(clippy::error_impl_error, reason = "common enough convention")]
pub enum Error {
    /// A command that we ran failed.
    CommandFailed(String, String, String, i32),

    /// Could not run a command.
    CommandRun(String, String, String, IoError),

    /// Could not read the configuration file.
    ConfigRead(IoError),

    /// Could not parse the configuration file.
    ConfigParse(AnyError),

    /// Could not parse the `mediaType` field in the configuration file.
    ConfigParseMediaType(MTVOwnedError),

    /// The configuration file declares an unsupported format version.
    ConfigUnsupportedVersion(u32, u32),

    /// Could not copy a directory.
    DirCopy(AnyError),

    /// Could not create a new directory.
    DirCreate(String, IoError),

    /// Could not determine a path's parent directory.
    DirParent(String),

    /// Something went really, really wrong.
    Internal(AnyError),

    /// The build step did not generate the expected executable file.
    NotBuilt(String, String),

    /// Some known missing functionality.
    NotImplemented(String),

    /// Could not look for a program in the search path.
    Which(AnyError),
}

impl Display for Error {
    #[inline]
    fn fmt(&self, f: &mut Formatter<'_>) -> FmtResult {
        match *self {
            Self::CommandFailed(ref program, ref action, ref command, ref rcode) => write!(
                f,
                "The `{command}` command to {action} {program} exited with code {rcode}"
            ),
            Self::CommandRun(ref program, ref action, ref command, _) => write!(
                f,
                "Could not run the `{command}` command to {action} {program}"
            ),
            Self::ConfigRead(_) => write!(f, "Could not read the configuration file"),
            Self::ConfigParse(_) => write!(f, "Could not parse the configuration file"),
            Self::ConfigParseMediaType(_) => write!(
                f,
                "Could not parse the mediaType field in the configuration file"
            ),
            Self::ConfigUnsupportedVersion(ref major, ref minor) => write!(
                f,
                "Unsupported configuration file format version {major}.{minor}"
            ),
            Self::DirCopy(_) => write!(f, "Could not copy a directory recursively"),
            Self::DirCreate(ref path, _) => write!(f, "Could not create the {path} directory"),
            Self::DirParent(ref path) => {
                write!(f, "Could not determine the parent directory of {path}")
            }
            Self::Internal(_) => write!(f, "check-build internal error"),
            Self::NotBuilt(ref program, ref name) => write!(
                f,
                "The build step for {program} did not generate the {name} executable file"
            ),
            Self::NotImplemented(ref msg) => write!(f, "Not implemented yet: {msg}"),
            Self::Which(_) => write!(f, "Could not look for a program in the search path"),
        }
    }
}

impl StdError for Error {
    #[inline]
    fn source(&self) -> Option<&(dyn StdError + 'static)> {
        match *self {
            Self::CommandFailed(_, _, _, _)
            | Self::ConfigUnsupportedVersion(_, _)
            | Self::DirParent(_)
            | Self::NotBuilt(_, _)
            | Self::NotImplemented(_) => None,

            Self::CommandRun(_, _, _, ref err)
            | Self::ConfigRead(ref err)
            | Self::DirCreate(_, ref err) => Some(err),

            Self::ConfigParse(ref err)
            | Self::DirCopy(ref err)
            | Self::Internal(ref err)
            | Self::Which(ref err) => Some(err.as_ref()),

            Self::ConfigParseMediaType(ref err) => Some(err),
        }
    }
}

/// Runtime configuration for the `check-build` routines.
#[derive(Debug)]
pub struct Config {
    /// The configuration file that describes the program to build and run.
    config: Utf8PathBuf,

    /// Verbose operation; display diagnostic output.
    verbose: bool,
}

impl Config {
    /// Create a [`Config`] object with the specified name.
    #[inline]
    #[must_use]
    pub const fn new(config: Utf8PathBuf) -> Self {
        Self {
            config,
            verbose: false,
        }
    }

    /// The configuration file that describes the program to build and run.
    #[inline]
    #[must_use]
    pub fn config(&self) -> &Utf8Path {
        &self.config
    }

    /// Verbose operation; display diagnostic output.
    #[inline]
    #[must_use]
    pub const fn verbose(&self) -> bool {
        self.verbose
    }

    /// Return a new [`Config`] object with the specified verbosity level.
    #[inline]
    #[must_use]
    pub fn with_verbose(self, verbose: bool) -> Self {
        Self { verbose, ..self }
    }
}
