// SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
// SPDX-License-Identifier: BSD-2-Clause
//! Prepare files, directories, and in-memory structures for building a program.

use camino::{Utf8Path, Utf8PathBuf};
use eyre::WrapErr as _;
use fs_extra::dir::{self, CopyOptions};
use log::debug;

use crate::defs::{Config, Error};
use crate::parse::Program;

#[cfg(test)]
use {crate::parse, eyre::eyre, std::fs};

/// A build environment for a single program.
#[derive(Debug)]
pub struct BuildEnv<'data> {
    /// The program name.
    name: &'data str,

    /// The program definition.
    prog: &'data Program,

    /// The temporary directory where we are building the program.
    tempd: &'data Utf8Path,

    /// The path where we copied the program sources over to.
    path: Utf8PathBuf,

    /// The directory where we copied the program source over from.
    srcdir: Utf8PathBuf,
}

impl BuildEnv<'_> {
    /// The program name.
    #[inline]
    #[must_use]
    pub const fn name(&self) -> &str {
        self.name
    }

    /// The program definition.
    #[inline]
    #[must_use]
    pub const fn prog(&self) -> &Program {
        self.prog
    }

    /// The temporary directory where we are building the program.
    #[inline]
    #[must_use]
    pub const fn tempd(&self) -> &Utf8Path {
        self.tempd
    }

    /// The path where we copied the program sources over to.
    #[inline]
    #[must_use]
    pub fn path(&self) -> &Utf8Path {
        &self.path
    }

    /// The directory where we copied the program source over from.
    #[inline]
    #[must_use]
    pub fn srcdir(&self) -> &Utf8Path {
        &self.srcdir
    }
}

/// Prepare the build environment for a single program.
///
/// # Errors
///
/// [`Error::DirParent`], [`Error::DirCreate`], [`Error::DirCopy`] on
/// filesystem operation failure.
#[inline]
pub fn build_env<'data>(
    cfg: &Config,
    name: &'data str,
    prog: &'data Program,
    tempd: &'data Utf8Path,
) -> Result<BuildEnv<'data>, Error> {
    let srcdir = cfg
        .config()
        .parent()
        .ok_or_else(|| Error::DirParent(cfg.config().to_string()))?
        .join(name);
    let path = tempd.join(name);
    debug!("Copying the source from {srcdir} to {path}");

    // We do expect the parent directory to exist
    dir::copy(
        &srcdir,
        &path,
        &CopyOptions {
            copy_inside: true,
            ..CopyOptions::default()
        },
    )
    .with_context(|| format!("Could not copy the {srcdir} directory to {path}"))
    .map_err(Error::DirCopy)?;
    Ok(BuildEnv {
        name,
        prog,
        tempd,
        path,
        srcdir,
    })
}

#[cfg(test)]
/// Prepare a build environment for running unit tests.
///
/// # Errors
///
/// Propagate errors from [`fs::create_dir_all`].
/// Fail if the specified name does not exist in the test program config.
#[inline]
pub fn build_test_build_env<PS>(name: &str, srcdir: PS) -> Result<(Config, &'static Program), Error>
where
    PS: AsRef<Utf8Path>,
{
    fs::create_dir_all(srcdir.as_ref().join(name))
        .with_context(|| format!("mkdir {srcdir}/{name}", srcdir = srcdir.as_ref()))
        .map_err(Error::Internal)?;

    let cfg = Config::new(srcdir.as_ref().join("config.toml")).with_verbose(true);
    let prog_config = parse::build_test_program_config();
    let Some(prog) = prog_config.program().get(name) else {
        return Err(Error::Internal(eyre!("No {name} in {prog_config:?}")));
    };
    Ok((cfg, prog))
}
